/*
 * Scone - The Web Enhancement Framework
 * Copyright (C) 2009 Harald Weinreich, Volkert Buchmann, Frank Wollenweber, Torsten Ha
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package scone;


import java.util.Vector;

import scone.proxy.Proxy;
import scone.util.PersistentProperties;
import scone.util.PersistentPropertiesGui;

import com.ibm.wbi.Meg;


/**
 * Classes implementing this interface may be registered as a plugin.
 *
 * <br>
 * Use <code>config/scone/plugins.xml</code> to enable and disable plugins.<br>
 * Use <code>config/properties/&lt;plugin-classname&gt;.xml</code> to configure plugins.<br>
 * At startup, the registered classes will be instantiated, stored in a Vector to
 * remain in memory and the init() method will be invoked.
 *
 * @author Harald Weinreich
 * @author Volkert Buchmann
 */
public abstract class Plugin {
    public static final String COPYRIGHT = "Harald Weinreich & Volkert Buchmann";

    private Proxy proxy = null;
    private Vector megs = new Vector();
    private PersistentProperties properties = null;

    /**
     * internal requirement! don't use it!
     */
    public final static int USEPARSINGMEG = 1 << 20;

    /**
     * possible requirements
     */
    public final static int NOREQUIREMENTS = 0;

    /**
     * fill HtmlDocumentTable and create HtmlDocument objects
     */
    public final static int PARSEDOCUMENT = 1 << 0 | USEPARSINGMEG;

    /**
     * fill LinkTable and create Link objects.<BR>
     * <B>Attention</B>: no LinkTokens will be created if CONSIDERLINKS is not required!
     */
    public final static int CONSIDERLINKS = 1 << 3 | USEPARSINGMEG;

    /**
     * fill InclusionTable and create Inclusion objects
     */
    public final static int CONSIDERINCLUSIONS = 1 << 4 | USEPARSINGMEG;

    /**
     * Get the Post-Data of the browser's request if available.<P>
     * The Data is made available by the PostDataExtractor and the ParsingMeg and stored in<BR>
     * <CODE>sconePipe.getTokenInputStream().getMetaInfo().get("postData")</CODE>
     */
    public final static int POSTDATA = 1 << 5 | USEPARSINGMEG;


    /**
     * fill keywordtable...
     */
    public final static int CONSIDERKEYWORDS = 1 << 6 | PARSEDOCUMENT;

    /**
     * Save the pure, unescaped text of the document body
     */
    public final static int SAVEBODYTEXT = 1 << 7 | PARSEDOCUMENT;

    /**
     * Save the source Code
     */
    public final static int SAVESOURCECODE = 1 << 8 | PARSEDOCUMENT;

    /**
     * Calculate a Fingerprint of the Web page source code.
     */
    public final static int CALCFINGERPRINT = 1 << 9 | PARSEDOCUMENT;

    /**
     * use AddPreambleEditor
     */
    public final static int ADDPREAMBLE = 1 << 16;

    /**
     * use GeneralResourceGenerator
     */
    public final static int RESOURCEGENERATOR = 1 << 17;

    /**
     * use NoCache - use to avoid caching in the browser...
     */
    public final static int NOCACHE = 1 << 18;

    /**
     * use PageTimout - use to limit caching in the browser...
     */
    public final static int PAGETIMEOUT = 1 << 19;

    /**
     * use RAS-Server (Remote Access Server)<BR>
     * The Plugin class has to implement ,,public int getRasPort()!
     */
    public final static int RAS = 1 << 20;

    /**
     * enable usertracking
     */
    public final static int ACCESSTRACKING = 1 << 24 | RESOURCEGENERATOR | RAS;  // SIMPLELINKS may be turned on in scone.accesstracking.AccessTracking

    /**
     * enable usertracking2: only with JavaScript but well working
     */
    public final static int ACCESSTRACKING2 = 1 << 25;

    /**
     * the standard condition for megs: !host=<Local scone URL> & content-type=text/html
     */
    public final static String HTDOCCONDITION = "!host=null & "
            + "!host=_* & !host=users.scone.de & !host=tracking.scone.de & "
            + "content-type=text/html";

    /**
     * condition for megs that handle NON-HTML-Documents: !host=<Local scone URL> & !content-type=text/html
     */
    public final static String NO_HTDOCCONDITION = "!host=null & "
            + "!host=_* & !host=users.scone.de & !host=tracking.scone.de & "
            + "!content-type=text/html";

    /**
     * condition for megs that parse any request/response with: !host=<Local scone URL>
     */
    public final static String DECENT_HOST_CONDITION = "!host=null & "
            + "!host=_* & !host=users.scone.de & !host=tracking.scone.de";


    // ----------------------------------------------------------------------

    /**
     * Define the requirements of this plugin by implementing this method.<BR>
     * A plugin has to specify its requirements which can be as follows:<BR>
     * NOREQUIREMENTS use this keyword if your plugin does not require any feature.<BR>
     * PARSEDOCUMENT if required, the HtmlDocument objects will be created
     *               and filled.<BR>
     * ACCESSTRACKING  if required, scone's usertracking feature is used<BR>
     * CONSIDERLINKS if required, Link Objects will be created and filled and
     *               LinkToken objects will be created<BR>
     * CONSIDERINCLUSIONS if required, INCLUSION objects will be created and filled.<BR>
     * CONSIDERKEYWORDS if required, Keyword objects will be created and filled.
     *                  This feature will automatically require PARSEDOCUMENT.<BR>
     * ADDPREAMBLE   causes the AddPreambleEditor to be added to the proxy.<BR>
     * RESOURCEGENERATOR causes the GeneralResourceGenerator to be added to the proxy<BR>
     * RAS The Remote Access Server is started.<P>
     *
     * These requirements have to be combined by the | operator:
     * PARSEDOCUMENT | CONSIDERLINKS will cause scone to create HtmlDocument objects,
     * Link objects and LinkToken objects.<P>
     *
     * If a feature is not required by any of the registered plugins, scone will not
     * support it at runtime.
     * @return the requirements
     */
    public abstract int getRequirements();

    /**
     * Specify the port of the RAS-Server. This has to be implemented for your
     * Plugin if you want to use the RAS-Server
     * @return The number of the port to be used by the Server.
     */
    public int getRasPort() {
        return 0;
    }

    /**
     * Initializes a scone plugin. This method is called by the <code>Scone</code> framework.
     */
    public abstract void init();

    /**
     * sets the proxy variable. This method is used by the Scone framework and
     * should not be overwritten.
     * @param proxy the Proxy object
     */
    public void init(Proxy proxy) {
        this.proxy = proxy;
    }

    /**
     * adds a MEG to WBI.
     * @param meg the MEG
     */
    public void addMeg(Meg meg) {
        proxy.addMeg(meg);
        megs.addElement(meg);
    }

    /**
     * returns a PersistentProperties object that the framework provides
     * for the plugin. It can be used conveniently to store properties.
     * It is automatically stored as <br>
     * config/properties/[full.classname].xml<br>
     * @return the PersistentProperties object for the plugin
     */
    public PersistentProperties getProperties() {
        if (properties == null) {
            properties = new PersistentProperties("config/properties/" + this.getClass().getName() + ".xml");
        }
        return properties;
    }

    /**
     * saves the properties if they have been changed.
     * This method is not called by the framework, strange...
     */
    public void saveProperties() {
        if (properties != null) {
            properties.store();
        }
    }

    /**
     * displays a GUI that allows for editing the persistent properties
     * for the plugin.
     * @param parent a parent Frame for the config dialog
     */
    public void displayConfigGUI(java.awt.Frame parent) {
        new PersistentPropertiesGui(parent, getProperties());
    }

    /**
     * displays a GUI that allows for editing the persistent properties
     * for the plugin.
     * @param parent a parent Dialog for the config dialog
     */
    public void displayConfigGUI(java.awt.Dialog parent) {
        new PersistentPropertiesGui(parent, getProperties());
    }

    /**
     * overwrite this method to enable plugin ectivities
     * not yet implemented!
     */
    public void enable() {}

    /**
     * overwrite this method to shutdown plugin activities
     * not yet implemented!
     */
    public void disable() {}

    /**
     * overwrite this method to be notified when the Proxy terminates
     */
    public void terminate() {}

}
