/*
 * Scone - The Web Enhancement Framework
 * Copyright (C) 2009 Harald Weinreich, Volkert Buchmann, Frank Wollenweber, Torsten Ha
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package scone.accesstracking;


// import java.io.*;
import java.util.Set;

import scone.netobjects.Access;


/**
 * This Class can be inherited by classes that want to be informed
 * on any change of open frames.
 *
 * @author Torsten Hass
 * @version 1.0b, 11/05/2002
 */

abstract public class FrameAccessObject {

    private FrameAccess fa = null;
    private Object notifyObject = null;
   
    /**
     * The constructor with out parameters gets the instance handle of FrameAccess, 
     * but does not register and ask to be called everytime a new frame opens or
     * an open frame closes. 
     */
    public FrameAccessObject() {
        fa = FrameAccess.getInstance();  // get instance handle of the one and only FrameAccess object
    }
      
    /**
     * The constructor with the register parameter gets the instance handle of FrameAccess and, 
     * depending on the value of <code>register</code>, registers at FrameAccess and asks to be 
     * informed everytime a new frame opens or an open frame closes. 
     * 
     * @param register      If true, the constructor registers at FrameAccess and 
     *                      and asks to be notified everytime a frames opens or closes.
     */
    public FrameAccessObject(boolean register) {
        fa = FrameAccess.getInstance();   // get instance handle of the one and only FrameAccess object
        if (register) {
            notifyObject = fa.notifyMe(this);
        }
    }   

    /**
     * notifyMe allows other objects to register for a callback on any change
     * of the FrameList and send the new Framelist.
     */
    public void notifyMe() {
        fa.notifyMe(this);
        this.notifyObject = this;
    }

    /**
     * stopNotifyMe clears the caller's reference from the list of regeistered
     * objects.
     * 
     */
    public void stopNotifyMe() {
        fa.stopNotifyMe(notifyObject);
    }

    /**
     * Returns true if a frame is open that matches the parentFrameName and frameName.
     *
     * @param parentframeName  The name of the parent frame
     * @param frameName        The String to be tested
     * @return                 True if the string exists.
     */
    public boolean contains(String parentFrameName,
            String frameName) {
        return fa.contains(parentFrameName, frameName);
    }

    /**
     * Returns true if a frame is open that matches the parentFrameAndFrameName.
     *
     * @param parentFrameAndFrameName  The name of the parent frame and the name of the frame concatinated
     * @return                 True if the string exists.
     */
    public boolean contains(String parentFrameAndFrameName) {
        return fa.contains(parentFrameAndFrameName);
    }

    /**
     * Gets the AppletConnector object for a specific frameName.
     * If there is no entry for frameName, null will be returned.
     *
     * @param parentframeName  The name of the parent frame
     * @param frameName        The name of the frame
     * @return                 The corresponding AppletConnector
     */
    public AppletConnector get(String parentFrameName,
            String frameName) {
        return fa.get(parentFrameName, frameName);
    }

    /**
     * Gets the AppletConnector object for a specific
     * parentFrameAndFrameName. This frameName is a concatination
     * of parentFrameName and frameName.
     * If the list contains no entry for the given frameName, null will be
     * returned.
     *
     * @param parentFrameAndFrameName   parentFrameName and frameName concatinated
     * @return                          The corresponding AppletConnector
     */
    public AppletConnector get(String parentFrameAndFrameName) {
        return fa.get(parentFrameAndFrameName);
    }

    /**
     * Gets the Access object for a specific frameName.
     * If there is no entry for frameName, null will be returned.
     *
     * @param parentframeName  The name of the parent frame
     * @param frameName        The name of the frame
     * @return                 The corresponding Access object
     */
    public Access getAccessObject(String parentFrameName,
            String frameName) {
        return fa.getAccessObject(parentFrameName, frameName);
    }

    /**
     * Gets the Access object for a specific
     * parentFrameAndFrameName. This frameName is a concatination
     * of parentFrameName and frameName.
     * If the list contains no entry for the given frameName, null will be
     * returned.
     *
     * @param parentFrameAndFrameName   parentFrameName and frameName concatinated
     * @return                          The corresponding Access object
     */
    public Access getAccessObject(String parentFrameAndFrameName) {
        return fa.getAccessObject(parentFrameAndFrameName);
    }

    /**
     * Show all parent frame names concatinated with the frame names in the list to stdout.
     */
    public void showOpenFrames() {
        fa.showOpenFrames();
    }

    /**
     * Returns a Set of frame names which are open right now.
     *
     * @return     A Set of parent frame names concatinated with frame names (strings) 
     */
    public Set getSetOfContents() {

        return fa.getSetOfContents();
    }

    /**
     * This abstract method has to be implemented by its subclass. It is called by FrameAccess
     * everytime a frame opens or closes.
     *
     * @param contents     A Set of string containing the parent frame names and frame names of the open frames
     */
    abstract public void framesChanged(Set contents);

}   

