/*
 * Scone - The Web Enhancement Framework
 * Copyright (C) 2009 Harald Weinreich, Volkert Buchmann, Frank Wollenweber, Torsten Ha
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
 package scone.netobjects;


import java.sql.ResultSet;


/**
 * An <code>Inclusion</code> represents the objects included in one web page, like applets, pictures etc.
 *
 * @author Harald Weinreich
 * @author Volkert Buchmann
 */

public class Inclusion extends TableRecord {
    public static final String COPYRIGHT = "(C) Harald Weinreich & Volkert Buchmann";

    protected static DBTableAdapter dbTable;

    protected NetNode parentNode = null;
    protected NetNode childNode = null;
   
    // set sql meta-data
    static {
        dbTable = new DBTableAdapter("InclusionTable"); 

        /*
         dbTable.addField("inclusionId",  dbTable.STRING,"",dbTable.KEY);
         dbTable.addField("parentNodeId", dbTable.STRING,"",dbTable.FIELD);
         dbTable.addField("childNodeId",  dbTable.STRING,"",dbTable.FIELD);
         dbTable.addField("tag",          dbTable.STRING,"",dbTable.FIELD);
         dbTable.addField("info",         dbTable.STRING,"",dbTable.FIELD);
         */
    }

    public Inclusion() {
        dbTable.init(this);           // Set values to default
    }

    /**
     * Create a new Inclusion object. The constructor tries to read a <code>Inclusion</code> set 
     * from the <code>scone</code> database, identified by the <code>inclusionId</code>. 
     * If such a set already exists, the values are read from the database. 
     * Else, hasRecordInDB is set to false
     * @param inclusionId the <code>id</code> of the inclusion.
     */   
    public Inclusion(String inclusionId) {
        dbTable.init(this);           // Set values to default
        fieldValues.put("inclusionId", inclusionId);
        dbTable.dbCheck(this);     // Read from DB
        if (hasRecordInDB) {                          // Object references are not filled by dbCheck.
            parentNode = NetNodeCache.getById(fieldValues.get("parentNodeId"));
            childNode = NetNodeCache.getById(fieldValues.get("childNodeId"));
        }  
    }

    /**
     * Creates a new Inclusion object. The constructor tries to read a <code>Inclusion</code> set 
     * from the <code>scone</code> database, this is identified by the 
     * <code>inclusionId</code>. 
     * If such a set already exists, the values are read from the database. 
     * Else, hasRecordInDB is set to false. Therfore <code>store()</code> creates OR updates the DB entry.
     * @param parent the <code>NetNode</code> of the document that embeds an other NetObject.
     * @param child the <code>NetNode</code> of the embedded object.
     */   
    // creates a new inclusion object
    protected Inclusion(NetNode parent, NetNode child) {
        dbTable.init(this);				      // DO NOT FORGET! Set values to default
        setInclusionId(SimpleUri.getHexHashCode("Inclsuion:" + parent.getNodeId() + child.getNodeId())); 
        dbTable.dbCheck(this);              // We have to check if it is in DB!
        if (hasRecordInDB)                          // Found in Database?
        {                                   // Object references are not filled by dbCheck.
            parentNode = NetNodeCache.getById(fieldValues.get("parentNodeId"));
            childNode = NetNodeCache.getById(fieldValues.get("childNodeId"));
        } else {
            // No: Try to fill myself...
            parentNode = parent;
            childNode = child;
            fieldValues.put("parentNodeId", parent.getNodeId());
            fieldValues.put("childNodeId", child.getNodeId());
        }
    }

    /**
     * Create Inclusion object to result of a DB-Query.
     * @param results the ResultSet returnetd by the query.
     */
    public Inclusion(ResultSet results) {
        dbTable.init(this);
        dbTable.fill(this, results);  // -> if there was a new row in results -> Fill user and hasRecordInDB=true
    }

    /**
     * returns the id of the Inclusion
     * @return the inclusionId
     */
    public String getInclusionId() {
        return fieldValues.get("inclusionId");
    }

    /**
     * sets the id of the Inclusion
     */
    private void setInclusionId(String id) {
        fieldValues.put("inclusionId", id);
    }

    /**
     * returns the id of the Node where the inclusion is defined
     * @return the nodeId of the parent node
     */
    public String getParentNodeId() {
        return fieldValues.get("parentNodeId");
    }
   
    /**
     * returns the Node where the inclusion is defined
     * @return the node where the inclusion is defined
     */
    public NetNode getParentNode() {
        return NetNodeCache.getById(getParentNodeId());
    }
   
    /**
     * returns the included nodeId
     * @return the inculded nodeId
     */
    public String getChildNodeId() {
        return fieldValues.get("childNodeId");
    }
   
    /**
     * returns the included node
     * @return the included node
     */
    public NetNode getChildNode() {
        return NetNodeCache.getById(getChildNodeId());
    }
   
    /**
     * returns the name of the tag which defined the inclusion
     * @return the name of the tag
     */
    public String getTag() {
        return fieldValues.get("tag");
    }
   
    /**
     * sets the name of the tag which defined the inclusion
     * @param tag the name of the tag
     */
    public void setTag(String tag) {
        fieldValues.put("tag", tag);
    }

    /**
     * returns some info to the tag. See Parsingmeg for more details...
     * @return info to the inclusion (like alt to image etc.)
     */
    public String getInfo() {
        return fieldValues.get("info");
    }
   
    /**
     * sets info to the tag which defined the inclusion
     * @param i info to the tag
     */
    public void setInfo(String i) {
        fieldValues.put("info", i);
    }

    /**
     * Store Inclusion to DB<BR>
     * For performance reasons, inclusion objects can be created without corresponding DB entry! 
     * Therefore store() can also create new DB entries. 
     */
    // save inclusion to DB	
    public void store() {
        if (hasRecordInDB) {    // Entry exists in DB: Update
            dbTable.updateDB(this);
        } else {
            dbTable.createInDB(this);
        }     // For perfomance reasons, inclusion objects are created in cache!
    }	
   
    /**
     * Returns the key(s)
     */
    public Object getKey(int KeyNo) {
        switch (KeyNo) {
        case 0:
            return getInclusionId();

        case 1:
            return InclusionCache.createKey(getParentNodeId(), getChildNodeId());

        default:
            return "";
        }
    }

}
