/*
 * Scone - The Web Enhancement Framework
 * Copyright (C) 2009 Harald Weinreich, Volkert Buchmann, Frank Wollenweber, Torsten Ha
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
 package scone.netobjects;


import java.sql.ResultSet;


/**
 * A <code>User</code> represents a Scone user.
 * One user can be associated to several <code>persons</code>.
 *
 * @author Harald Weinreich
 * @author Volkert Buchmann
 */

public class User extends TableRecord {
    public static final String COPYRIGHT = "(C) Harald Weinreich & Volkert Buchmann";

    protected static DBTableAdapter dbTable;
    protected Person person = null;

    // set sql meta-data
    static {
        dbTable = new DBTableAdapter("UserTable");

        /*
         dbTable.addField("userId",       dbTable.NUMBER,"-1", dbTable.GENERATED);
         dbTable.addField("userName",     dbTable.STRING,"",   dbTable.KEY);
         dbTable.addField("userPass",     dbTable.STRING,"",   dbTable.FIELD);
         dbTable.addField("personId",     dbTable.STRING,"",   dbTable.FIELD);
         dbTable.addField("lastLogin",    dbTable.NUMBER,"0",  dbTable.FIELD);
         */
    }

    /**
     * returns the id of that user
     * @return userId
     */
    public String getUserId() {
        return fieldValues.get("userId");
    }

    /**
     * returns the name of the user
     * @return userName
     */
    public String getUserName() {
        return fieldValues.get("userName");
    }

    /**
     * returns the personId to the user: One person can have several accounts.
     * @return personId
     */
    public String getPersonId() {
        return fieldValues.get("personId");
    }

    /**
     * sets the personId to the user: One person can have several accounts.
     * @param value the personId
     */
    public void setPersonId(String value) {
        fieldValues.put("personId", value);
    }

    /**
     * returns the person data to this user
     * @return person
     */
    public Person getPerson() {
        if (person == null) {
            person = PersonCache.getById(getPersonId());
        }
        return person;
    }

    /**
     * returns the password of the user
     * @return userPass
     */
    public String getUserPass() {
        return fieldValues.get("userPass");
    }

    /**
     * sets the password of the user
     * @param value the password
     */
    public void setUserPass(String value) {
        fieldValues.put("userPass", value);
    }

    public String getLastLogin() {
        return fieldValues.get("lastLogin");
    }

    public void setLastLogin(String value) {
        fieldValues.put("lastLogin", value);

    }

    /**
     * Check if a user to a name exists in the db yet.
     * @param name the name of the user
     * @param create create in DB if the user does not yet exist. false: check only
     */
    public User(String name, boolean create) {
        dbTable.init(this);
        fieldValues.put("userName", name);
        if (create) {
            dbTable.dbInit(this);
        } else {
            dbTable.dbCheck(this);
        }     // just check the database for a set
    }

    /**
     * Create a new user object
     */
    public User(String name) {
        this(name, true);
    }

    /**
     * Create User to result of DB query.
     * @param results the resultset returned by the query
     */
    public User(ResultSet results) {
        dbTable.init(this);
        dbTable.fill(this, results);  // -> if there was a new row in results -> Fill user and hasRecordInDB=true
    }

    /**
     * Check if a user to an id exists in the db.
     * @param id the id of the user
     * @param create create in DB if the user does not yet exist. false: check only
     */
    public User(long id, boolean create) {
        dbTable.init(this);
        fieldValues.put("userId", String.valueOf(id));
        if (create) {
            dbTable.dbInit(this);
        } else {
            dbTable.dbCheck(this);
        }     // just check the database for a set
    }

    /**
     * Create a new user object with a name if it doesn't exist in the db yet.
     * @param name the name of the user
     */
    public User(long id) {
        this(id, true);
    }

    /**
     * store object to database.
     */
    public void store() {
        dbTable.updateDB(this);
    }

    public Object getKey(int KeyNo) {
        switch (KeyNo) {
        case 0:
            return getUserId();

        case 1:
            return getUserName();

        default:
            return "";
        }
    }

}
