/*
 * Scone - The Web Enhancement Framework
 * Copyright (C) 2009 Harald Weinreich, Volkert Buchmann, Frank Wollenweber, Torsten Ha
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package scone.proxy;


import scone.util.ErrorLog;
import scone.util.tokenstream.HtmlTagToken;
import scone.util.tokenstream.HtmlTextToken;
import scone.util.tokenstream.SconePipe;
import scone.util.tokenstream.Token;
import scone.util.tokenstream.TokenInputStream;
import scone.util.tokenstream.TokenOutputStream;


/**
 * Adds a preamble to a Html-Document.
 *
 * <br>
 * The preamble is written before the &lt;body&gt;-Tag, or before the document body begins.<p>
 *
 * usage<p>
 * <code><pre> preambleAdder = new AddPreambleHtmlTokenHandler();
 * preambleAdder.addPreambleString("foo");
 * controller.addMonitor(preambleAdder);</pre></code>
 *
 * @author Harald Weinreich
 * @author Volkert Buchmann
 */

public class AddPreambleEditor extends HtmlTokenEditor {

    public static final String COPYRIGHT = "(C) Harald Weinreich & Volkert Buchmann";

    public static HtmlTextToken preamble = new HtmlTextToken("");

    /**
     * add a String to preamble
     *
     * @param preambleString the String which will be inserted.
     */
    public static void addPreambleString(String preambleString) {
        preamble.appendText(preambleString);
    }

    /**
     * tries to add the preamble:
     * <OL>
     * <LI> before &lt;/head&gt;-Tag
     * <LI> before &lt;body&gt;-Tag
     * <LI> before &lt;frameset&gt;-Tag
     * <LI> before heading
     * <LI> before table
     * <LI> before link (&lt;A&gt; or &lt;AREA&gt;-Tag
     * <LI> before image
     * <LI> before form
     * <LI> before preformatted text
     * </OL>
     */

    public void handleRequest(SconePipe pipe) {
        try {
            TokenInputStream in = pipe.getTokenInputStream();
            TokenOutputStream out = pipe.getTokenOutputStream();

            boolean written = false;
            Token t = null;
            HtmlTagToken tag = null;

            while ((t = in.read()) != null) {

                if (!written && (t instanceof HtmlTagToken)) {
                    tag = (HtmlTagToken) t;

                    if (
                                    ((tag.getTagType() == tag.T_HEAD)
                                    && (tag.isEndTag()))
                            || (tag.getTagType() == tag.T_BODY)
                            ||    // the body is about to start...
                                    (tag.getTagType() == tag.T_FRAMESET)
                            ||// Frames?
                                    tag.isInBody())      // in body?
                    {
                        preamble.setText("\n" + preamble.getText() + "\n");
                        out.write(preamble);   // write preamble
                        written = true;
                    }
                }
                out.write(t);
            }
        } catch (Exception exc) {
            ErrorLog.log(this, "handleRequest()", "", exc);
        }
    }

}

