/*
 * Scone - The Web Enhancement Framework
 * Copyright (C) 2009 Harald Weinreich, Volkert Buchmann, Frank Wollenweber, Torsten Ha
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package scone.proxy;


import java.io.IOException;

import scone.util.ErrorLog;
import scone.util.tokenstream.SconePipe;

import com.ibm.wbi.RequestEvent;
import com.ibm.wbi.RequestRejectedException;
import com.ibm.wbi.protocol.http.DocumentInfo;
import com.ibm.wbi.protocol.http.HttpEditor;
import com.ibm.wbi.protocol.http.HttpResponseHeader;


/**
 * The HtmlTokenEditor is a WBI MEG that allows for manipulating
 * HTML documents in the HTTP-Response of the proxy.
 * <br>
 *
 * It may be declared like this:<br>
 *
 * MyHtmlTokenEditor mhte = new MyHtmlTokenEditor();<br>
 * mhte.setup(HTDOCCONDITION,priority);<br>
 * addMeg(mhte);<br>
 *
 * Now MyHtmlTokenEditor's handleRequest(SconePipe) method will be called
 * whenever the proxy sends an HTML document to the client.
 */

public abstract class HtmlTokenEditor extends HttpEditor {
    public static final String COPYRIGHT = "(C) Harald Weinreich & Volkert Buchmann";

    protected RequestEvent e = null;

    /**
     * the MEG's handleRequest() method. Do not use it.
     */
    public final void handleRequest(RequestEvent e) throws RequestRejectedException, IOException {
        SconePipe pipe = null;

        this.e = e;
        try {
            // create pipe...
            pipe = new SconePipe(e);
            handleRequest(pipe);
            pipe.close();
        } catch (Exception exc) {
            ErrorLog.log(this, "handleRequest()", "", exc);
        } finally {
            try {
                pipe.close();
            } catch (Exception exc2) {
                exc2.printStackTrace();
            }
        }
    }

    /**
     * this method adds fields to the HTTP response header.
     * It must be called before any of the streams is requested from the pipe!
     * @param name the field's name
     * @param value the value
     */

    public void setResponseHeaderField(String name, String value) {
        DocumentInfo documentInfo = (DocumentInfo) e.getRequestInfo();
        HttpResponseHeader hrh = documentInfo.getHttpResponseHeader();

        hrh.set(name, value);
    }

    /**
     * overwrite this method to use the HtmlTokenEditor. This method is called every
     * time an HTML document is sent to the client. The SconePipe will contain the document.
     * Note: The tokens fron the TokenInputStream must be written into the
     * TokenOutputStream.
     * @param pipe the SconePipe containing the document.
     */
    public abstract void handleRequest(SconePipe pipe);
}
