/*
 * Scone - The Web Enhancement Framework
 * Copyright (C) 2009 Harald Weinreich, Volkert Buchmann, Frank Wollenweber, Torsten Ha
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
 package scone.robot;

/**
* The FairBreadthSearchURLQueue simulates a set of fifo-ques, one for every robotTask.
* In every dequeue operation an entry from a different task is taken form the queue. A round-robin strategy is used to choose the next task.
*
* @author Frank Wollenweber
*/


public class FairBreadthSearchURLQueue extends GenericURLQueue {

    protected Ring ring; // The robotTasks of the entries in the queue are stored in this data-structure


    /**
    * constructor
    **/
    public FairBreadthSearchURLQueue() {
        super();
        ring = new Ring();
    }

    /**
    * Get the next entry from the queue
    * @return The next entry
    */
    public synchronized QueueEntry dequeue() {
        RobotTask r = (RobotTask)ring.getNext();
        QueueEntry qe = null;
        for(int i = 0;i < queue.size();i++) {
            qe = (QueueEntry)queue.elementAt(i);
            if(qe.getRobotTask() == r) {
                queue.removeElementAt(i);
                if(getNumberOfPendingQueueEntries(qe.getRobotTask()) == 0) {
                    ring.remove(qe.getRobotTask());
                }
                numberOfElements--;
                return qe;
            }
        }
        return null;
    }


    /**
    * Remove all queueEntries of the specified robotTask
    * @param robotTask removes all queue entries of this task
    */
    public synchronized void removeAllQueueEntries(RobotTask robotTask) {
        super.removeAllQueueEntries(robotTask);
        ring.remove(robotTask);
    }


    /**
    * Remove an entry from the queue
    * @param qe The entry to remove
    */
    public synchronized void removeQueueEntry(QueueEntry qe) {
        super.removeQueueEntry(qe);
        if(getNumberOfPendingQueueEntries(qe.getRobotTask()) == 0) {
            ring.remove(qe.getRobotTask());
        }
    }



    /**
    * This class implements an element in the ring data-structure
    **/
    private class RingElement {

        private Object object; // Reference to the object at this position of the ring
        private RingElement nextElement; // The next RingElement


        /**
        * constructur
        * @ param object Store this object in the ring
        **/
        RingElement(Object object) {
            this.object = object;
        }


        /**
        * Get the object stored at this RingElement
        * @return The Object
        **/
        Object getObject() {
            return object;
        }

        /**
        * Get the next element
        * @return Next element
        **/
        RingElement getNext() {
            return nextElement;
        }

        /**
        * Sets the next element
        * @param r The next RingElement
        **/
        void setNext(RingElement r) {
            nextElement = r;
        }
    }


    /**
    * This class implements a ring data-structure. It is similar to a linear list, but there's a pointer from the last element to the first
    **/

    private class Ring {

        private RingElement nextElement; // In every getNext operation this is set to the next element in the ring
        private RingElement lastElement; // The last element of the ring


        /**
        * Adds an element to the ring. An object is only stored once in the ring-structure.
        * @param object Add this object
        **/
        synchronized void add(Object object) {
            RingElement r;
            boolean found = false;
            if(lastElement != null) {
                RingElement rStart = lastElement.getNext();
                RingElement re = rStart;
                do {
                    if(re.getObject() == object) {
                        found = true;
                    }
                    else {
                        re = re.getNext();
                    }
                }while(!((found) || (re == rStart)));
                if(!found) {
                    r = new RingElement(object);
                    r.setNext(lastElement.getNext());
                    lastElement.setNext(r);
                    lastElement = r;
                }
            }
            else {
                r = new RingElement(object);
                lastElement = r;
                lastElement.setNext(r);
                nextElement = r;
            }
        }

        /**
        * Removes object from the ring
        * @param object Remove this object
        **/
        synchronized void remove(Object object) {
            if(lastElement != null) {
                RingElement rStart = lastElement.getNext();
                RingElement r = rStart;
                RingElement rp = lastElement;
                do {
                    if(r.getObject() == object) {

                        //remove r from list
                        if(rp != r) {
                            if(r == nextElement) {
                                nextElement = r.getNext();
                            }
                            if(r == lastElement) {
                                lastElement = r.getNext();
                            }
                            rp.setNext(r.getNext());
                            r = r.getNext();
                        }
                        else {
                            lastElement = null;
                            nextElement = null;
                        }
                    }
                    else {
                        rp = r;
                        r = r.getNext();
                    }
                }while(!((lastElement == null) || (r == rStart)));
            }
        }

        /**
        * Get the next object from the ring.
        * @return The next object
        **/
        synchronized Object getNext() {
            if(nextElement != null) {
                Object o = nextElement.getObject();
                nextElement = nextElement.getNext();
                return o;
            }
            else {
                return null;
            }
        }
    }
}