/*
 * Scone - The Web Enhancement Framework
 * Copyright (C) 2009 Harald Weinreich, Volkert Buchmann, Frank Wollenweber, Torsten Ha
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package scone.robot;

import java.util.Enumeration;
import java.util.Vector;

import scone.netobjects.SimpleUri;

/**
* Implementation of a generic queue
*
* @author Frank Wollenweber
*/



public abstract class GenericURLQueue implements URLQueue {

    protected Vector queue; // A vector is used for the queue
    protected int numberOfElements; // Number of entries


    /**
    * The constructor
    */
    public GenericURLQueue() {
        queue = new Vector();
        numberOfElements = 0;
    }


    /**
    * Adds a new entry to the queue
    * @param qe The new entry
    */
    public synchronized void queue(QueueEntry qe) {
        queue.add(qe);
        numberOfElements++;
    }


    /**
    * Get the next entry from the queue
    * @return The next entry
    */
    //public abstract synchronized QueueEntry dequeue()


    /**
    * Get the number of entries waiting in the queue
    * @return Number of elements
    */
     public synchronized int size() {
        return queue.size();
    }


    /**
    * Get the number of entries belonging to the robotTask
    * @param robotTask only entries of this task are counted
    * @return Number of entries
    */
    public synchronized int getNumberOfPendingQueueEntries(RobotTask robotTask) {
        int numberOfPendingQueueEntries = 0;
        Enumeration queueEntries = queue.elements();
        while(queueEntries.hasMoreElements()) {
            QueueEntry qe = (QueueEntry)queueEntries.nextElement();
            RobotTask rt = qe.getRobotTask();
            if(rt.equals(robotTask)) {
                numberOfPendingQueueEntries++;
            }
        }
        return numberOfPendingQueueEntries;
    }


    /**
    * Get vector with all entries belonging to the robotTask
    * @param robotTask only entries of this task are added to the vector
    * @return Vector with queueEntries
    */
    public synchronized Vector getPendingQueueEntries(RobotTask robotTask) {
        Vector pendingQueueEntries = new Vector();
        Enumeration queueEntries = queue.elements();
        while(queueEntries.hasMoreElements()) {
            QueueEntry qe = (QueueEntry)queueEntries.nextElement();
            RobotTask rt = qe.getRobotTask();
            if(rt.equals(robotTask)) {
                pendingQueueEntries.add(qe);
            }
        }
        return pendingQueueEntries;
    }


    /**
    * Remove all queueEntries of the specified robotTask
    * @param robotTask removes all queue entries of this task
    */
    public synchronized void removeAllQueueEntries(RobotTask robotTask) {
        Vector pendingQueueEntries = getPendingQueueEntries(robotTask);
        Enumeration queueEntries = pendingQueueEntries.elements();
        while(queueEntries.hasMoreElements()) {
            QueueEntry qe = (QueueEntry)queueEntries.nextElement();
            queue.remove(qe);
        }
    }


    /**
    * The number of queueEntries from different tasks waiting to download this url
    * @param url Only queueEntries with this url are counted
    * @return Number of entries
    */
    public synchronized int getNumberOfPendingQueueEntries(SimpleUri uri) {
        int numberOfPendingQueueEntries = 0;
        Enumeration queueEntries = queue.elements();
        while(queueEntries.hasMoreElements()) {
            QueueEntry qe = (QueueEntry)queueEntries.nextElement();
            SimpleUri link = qe.getURI();
            if(link.equals(uri)) {
                numberOfPendingQueueEntries++;
            }
        }
        return numberOfPendingQueueEntries;
    }

    /**
    * The vector with all queueEntries from different tasks waiting to download this url
    * @param url Only queueEntries with this url are added to the vector
    * @return Vector with queueEntries
    */
    public synchronized Vector getPendingQueueEntries(SimpleUri uri) {
        Vector pendingQueueEntries = new Vector();
        Enumeration queueEntries = queue.elements();
        while(queueEntries.hasMoreElements()) {
            QueueEntry qe = (QueueEntry)queueEntries.nextElement();
            SimpleUri link = qe.getURI();
            if(link.equals(uri)) {
                pendingQueueEntries.add(qe);
            }
        }
        return pendingQueueEntries;
    }

    /**
    * Checks, if there is an entry belonging to robotTask, which is waiting to download url
    * @param url The url to check
    * @param robotTask Only entries of this task are checked
    * @return True, if there is an entry waiting
    */
    public synchronized boolean isPendingURL(SimpleUri uri, RobotTask robotTask) {
        Enumeration queueEntries = queue.elements();
        while(queueEntries.hasMoreElements()) {
            QueueEntry qe = (QueueEntry)queueEntries.nextElement();
            SimpleUri link = qe.getURI();
            if((link.equals(uri)) && (qe.getRobotTask().equals(robotTask))) {
                return true;
            }
        }
        return false;
    }


    /**
    * Get the entry belonging to robotTask and waiting to download url
    * @param url The url to check
    * @param robotTask Only entries of this task are checked
    * @return The waiting queueEntry
    */
    public synchronized QueueEntry getPendingURL(SimpleUri uri, RobotTask robotTask) {
        Enumeration queueEntries = queue.elements();
        while(queueEntries.hasMoreElements()) {
            QueueEntry qe = (QueueEntry)queueEntries.nextElement();
            SimpleUri link = qe.getURI();
            if((link.equals(uri)) && (qe.getRobotTask().equals(robotTask))) {
                return qe;
            }
        }
        return null;
    }

    /**
    * Remove an entry from the queue
    * @param qe The entry to remove
    */
    public synchronized void removeQueueEntry(QueueEntry qe) {
        queue.remove(qe);
    }
}
