/*
 * Scone - The Web Enhancement Framework
 * Copyright (C) 2009 Harald Weinreich, Volkert Buchmann, Frank Wollenweber, Torsten Ha
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package scone.robot;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;

import scone.util.ErrorLog;


/**
 * This input counts the bytes and can be limited to read only a certain number of bytes
 *
 * @author      Frank Wollenweber
 */

class LimitedInputStream extends FilterInputStream {

    private int counter = 0; // the byte counter
    private int limit = -1; // -1 means no limit
    private boolean limitReached = false; // This value indicates if the limit was reached

    /**
     * constructor
     * @param in The input stream to make a limitedInputstream of
     */
    public LimitedInputStream(InputStream in) {
        super(in);
    }

    /**
     * constructor
     * @param in The input stream to make a limitedInputstream of
     * @param limit The stream reads only limit bytes
     */
    public LimitedInputStream(InputStream in, int limit) {
        super(in);
        this.limit = limit;
    }

    /**
     * Reads the next byte of data from the input stream. <br>
     * The value byte is returned as an int in the range 0 to 255.<br>
     * If no byte is available because the end of the stream has been reached or the limit has been reached, the value -1 is returned.
     * @return the value of the next byte or -1 if the end or the limit of the stream has been reached.
     */
    public int read() throws IOException {
        if (counter >= limit) {
            limitReached = true;
            return -1;
        }
        int b = in.read();

        if (b != -1) {
            counter++;
        }
        return b;
    }

    /**
     * Reads some number of bytes from the input stream and stores them into the buffer array buffer.
     * The number of bytes actually read is returned as an integer.
     * @param buffer the buffer into which the data is read.
     * @param offset the offset for the beginning of the read operation
     * @param length read lenght bytes
     * @return Number of bytes read into the buffer or -1 if the end of the stream or the limit has been reached.
     */
    public int read(byte[] buffer, int offset, int length) throws IOException {
        if (counter >= limit) {
            limitReached = true;
            return -1;
        }
        if ((counter + length) > limit) {
            length = limit - counter;
        }

        try{
            int bytesRead = in.read(buffer, offset, length);
            if (bytesRead != -1) {
                counter += bytesRead;
            }
            return bytesRead;
        } catch (Exception e) {
            System.out.println("LimitedInputStream: Read timed out!");
            ErrorLog.log(this, "LimitedInputStream", "Read timed out!", e);
        }
        return -1;
    }

    /**
     * Reads bytes from the input stream and stores them into the buffer array b.
     * @param b the buffer into which the data is read. b.length bytes are read.
     * @return Number of bytes read into the buffer or -1 if the end of the stream or the limit has been reached.
     **/
    public int read(byte[] b) throws IOException {
        return read(b, 0, b.length);
    }

    /**
     * Return number of bytes read so far.
     * @return Number of bytes
     */
    public int numberOfBytesRead() {
        return counter;
    }

    /**
     * Checks whether the size-limit has been reached during read operations
     * @return true, if the limit has been reached
     **/
    public boolean getLimitReached() {
        return limitReached;
    }
}