/*
 * Scone - The Web Enhancement Framework
 * Copyright (C) 2009 Harald Weinreich, Volkert Buchmann, Frank Wollenweber, Torsten Ha
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
 package scone.robot;


import java.util.Hashtable;

import scone.netobjects.HtmlNode;
import scone.netobjects.HtmlNodeCache;
import scone.netobjects.NetNode;
import scone.netobjects.NetNodeCache;
import scone.netobjects.SimpleUri;


/**
 * A robotHtmlNode extends the concept of NetNodes and HtmlNodes to fit the robot's needs. It has a set of attributes and a depth value.
 * There can be many RobotHtmlNodes for one NetNode
 *
 * @author Frank Wollenweber
 */



public class RobotHtmlNode {

    private String nodeId; // Many robotHtmlNodes can reference the same NetNode
    private SimpleUri simpleUri; // The URI of this NetNode. It is redundant but useful for a better performance
    private int scannedDepth; // The scanned depth of this node
    private Hashtable pageAttributes; // Classifier can add attributes to this object
    private boolean onlyNetNode; // This is true, if no HtmlNode belongs to this RobotHtmlNode

    /**
     * Constructor
     * @param htmlNode the htmlNode this robotHtmlNode belongs to
     * @param depth this node was scanned with this depth
     */
    public RobotHtmlNode(HtmlNode htmlNode, int scannedDepth) {
        nodeId = htmlNode.getNode().getNodeId();
        simpleUri = htmlNode.getNode().getSUri();
        this.scannedDepth = scannedDepth;
        pageAttributes = new Hashtable();
        onlyNetNode = false;
    }

    /**
     * Constructor
     * @param netNode the netNode this robotHtmlNode belongs to. Might be used for documents with other mime-types then text/html
     * @param depth this node was scanned with this depth
     */
    public RobotHtmlNode(NetNode netNode, int scannedDepth) {
        nodeId = netNode.getNodeId();
        simpleUri = netNode.getSUri();
        this.scannedDepth = scannedDepth;
        pageAttributes = new Hashtable();
        onlyNetNode = true;
    }

    /**
     * Get the simpleUri of this Node
     * @return the simpleUri
     */
    public SimpleUri getSimpleUri() {
        return simpleUri;
    }

    /**
     * Get the htmlNode if existant. <b>This may throw a NullPointerException</b>
     * @return the htmlNode
     */
    public HtmlNode getHtmlNode() throws NoHtmlNodeException {
        if (onlyNetNode) {
            throw new NoHtmlNodeException();
        }
        return HtmlNodeCache.get(NetNodeCache.getById(nodeId));
    }

    /**
     * Set the htmlNode
     * @param htmlNode the htmlNode
     */
    public void setHtmlNode(HtmlNode htmlNode) {
        nodeId = htmlNode.getNode().getNodeId();
        simpleUri = htmlNode.getNode().getSUri();
        onlyNetNode = false;
    }

    /**
     * Get the netNode
     * @return the netNode
     */
    public NetNode getNetNode() {
        return NetNodeCache.getById(nodeId);
    }

    /**
     * Get the ID of this robotHtmlNode
     * @return the nodeId
     */
    public String getNodeId() {
        return nodeId;
    }

    /**
     * Set the netNode
     * @param netNode the netNode
     */
    public void setNetNode(NetNode netNode) {
        nodeId = netNode.getNodeId();
        simpleUri = netNode.getSUri();
        onlyNetNode = true;
    }

    /**
     * Get the scanned depth of this node
     * @return the scanned depth
     */
    public int getScannedDepth() {
        return scannedDepth;
    }

    /**
     * Set the scanned depth
     * @param scanned depth the depth
     */
    public void setScannedDepth(int scannedDepth) {
        this.scannedDepth = scannedDepth;
    }

    /**
     * Get the value of the specified attribute
     * @param key This key identifies one attribute
     * @return The value of the attribute
     */
    public String getAttribute(String key) {
        return (String) pageAttributes.get(key);
    }

    /**
     * Set the value of the specified attribute
     * @param key The name of the attribute
     * @param value The value of the attribute
     */
    public void setAttribute(String key, String value) {
        pageAttributes.put(key, value);
    }
    
}
