/*
 * Scone - The Web Enhancement Framework
 * Copyright (C) 2009 Harald Weinreich, Volkert Buchmann, Frank Wollenweber, Torsten Ha
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
 package scone.robot;


import java.util.Enumeration;
import java.util.Observable;
import java.util.Observer;

import scone.Plugin;
import scone.netobjects.AccessCache;
import scone.netobjects.AccessEvent;
import scone.netobjects.HtmlNode;
import scone.netobjects.HtmlNodeCache;


/**
 * This plugin tests the robot's basic functionality. Register it as a scone plugin. It will crawl the link-environment of every page displayed in the browser.
 *
 * @author Frank Wollenweber, Harald Weinreich
 */



public class RobotTestPlugin extends Plugin implements Observer, RobotUser {

    private Robot robot;
    private RobotTask rt;
    private int depth = 2; // Crawl depth of Plugin -> read from properties.xml
    private int restriction = RobotTask.INTERNAL; // Crawl restriction...

    /**
     * constructor
     */
    public RobotTestPlugin() {}

    /**
     * requirements
     * @return this task's requirements
     */
    public int getRequirements() {
        return PARSEDOCUMENT | ACCESSTRACKING | CONSIDERLINKS
                | CONSIDERINCLUSIONS;
    }

    /**
     * Scone calls this method from all plugins. It reads the configuration from scone.robot.RobotTestPlugin.xml
     */
    public void init() {
        robot = Robot.instance();
        AccessCache.putObserver(this);

        // Read the "Crawl depth"...
        if (getProperties().get("Crawl depth") != null) {
            int d = Integer.parseInt(getProperties().get("Crawl depth"));

            if ((d < 0) || (d > 10)) { // Other numbers are quite stupid...
                throw new NumberFormatException();
            } else {
                depth = d;
                System.out.print("(" + depth + ")");
            }
        } else {
            throw new NumberFormatException();
        }

        // Read the crawl method: "Crawl restrictions" from XML-File
        if (getProperties().get("Crawl restriction") != null) {
            System.out.print("(" + getProperties().get("Crawl restriction") + ")");
            if (getProperties().get("Crawl restriction").equals("Internal")) {
                restriction = RobotTask.INTERNAL;
            } else {
                if (getProperties().get("Crawl restriction").equals("Sub-directories")) {
                    restriction = RobotTask.SUBDIRECTORIES;
                } else {
                    if (getProperties().get("Crawl restriction").equals("External")) {
                        restriction = RobotTask.EXTERNAL;
                    } else {
                        if (getProperties().get("Crawl restriction").equals("No restriction")) {
                            restriction = RobotTask.ALL;
                        } else {
                            throw new NumberFormatException();
                        }
                    }
                }
            }
        }
        System.out.print("...");
    }

    /**
     * New Scone event
     */
    public void update(Observable o, Object arg) {
        if (o instanceof AccessCache) {
            AccessEvent e = (AccessEvent) arg;

            if (e.getAccess().getAction() == 0
                    && e.getAccess().getStayTime() == 0) { // first Access-Event!
                System.out.println("RobotTest ---> " + e.getUser().getUserName() + " accessed " + e.getNode().toString());
                HtmlNode hNode = HtmlNodeCache.check(e.getNode());

                if (hNode == null) {
                    System.out.println("Error: Access from non HTML-Page!");
                } else {
                    try {
                        if (rt != null) {
                            robot.stopRobotTask(rt);
                        }
                        rt = new RobotTask(hNode.getSUri(), depth, restriction, this);
                        robot.scan(rt);
                    } catch (Exception ex) {}
                    System.out.println("Title: " + hNode.getTitle());
                }
            }
        }
    }

    /**
     * The robot found a new document
     * @param robotHtmlNode the new document
     * @param robotTask identifies the task which causes the robot to download robotHtmlNode.
     */
    public void robotNewPage(RobotHtmlNode robotHtmlNode, RobotTask robotTask) {}

    /**
     * The robot finished the robotTask
     * @param robotTask the robot finished this task
     */
    public void robotTaskFinished(RobotTask robotTask) {
        System.out.println("Task: " + robotTask.getStartURI().toString() + " Depth: " + robotTask.getDepth() + " finished");
        System.out.println("Results: ");
        Enumeration downloadedNodes = robotTask.getResultNodes();

        while (downloadedNodes.hasMoreElements()) {
            RobotHtmlNode robotHtmlNode = (RobotHtmlNode) downloadedNodes.nextElement();

            System.out.println(robotHtmlNode.getSimpleUri().toString());
        }
    }
}
