/*
 * Scone - The Web Enhancement Framework
 * Copyright (c) 2004 Harald Weinreich, Volkert Buchmann, Frank Wollenweber, Torsten Ha
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package scone.usertesttool;


import java.util.Set;

import scone.util.ErrorLog;



/**
 * This object keeps up to date about opened and closed
 * web pages 
 
 * @author Torsten Hass
 * @version 1.1, 11/12/2003
 */
public class BrowserControl extends scone.accesstracking.FrameAccessObject {

    private Set openFrames = null; // holds a list of frame names currently open
    
    public BrowserControl () {

        notifyMe();               // inform my method FramesChanged() about opening and closing web pages
//        System.out.println("BrowserControl: gestartet und angemeldet");
    }

    /**
     * This method is called by FrameAccess everytime a new frame was added to
     * or removed from list of active frames. 
     *
     * @param contents A Set of parent frame names (if existing) concated with
     * frame names
     */ 
    public void framesChanged(Set contents) {

        openFrames = contents;
    }

    /**
     * This method is called by UserTestControl to open a new uri in the browser
     *
     * @param parentFrameAndFrameName The name of the parent frame and current frame concatinated to identify the browser window
     * @param uri The uri to be opened
     * @return True, if no problems occured
     */
    public boolean openUri(String parentFrameAndFrameName, String uri) {
        scone.accesstracking.AppletConnector appletConnector = null;
        boolean success = false;
        
        // get the AppletConnector object
        try {
            parentFrameAndFrameName = parentFrameAndFrameName.trim();
        } catch (NullPointerException e) {
            ErrorLog.log(this, "openUri", "The string \"parentFrameAndFrameName\" is null", e);
        }
        try {
            appletConnector = get (parentFrameAndFrameName);
            if (appletConnector != null) {
                // open a new uri in browser
                appletConnector.sendToApplet("OpenURLDelayed", uri, "_self");   // open new URL in same browser window
                success = true;
                return success;
            }
        } catch (Exception e) {
            ErrorLog.log(this, "openUri", "Tried to open an uri, but the AppletConnector was null", e);
        }
        return false;

    }
    

    /**
     * This method calls JavaScript code inside the browser using the Applet to bring the browser's window to front
     *
     * @param parentFrameAndFrameName The name of the browserframe and, if existing, the framename of the parent frame to identify the right browser
     */
    public boolean bringToFront(String parentFrameAndFrameName) {
        scone.accesstracking.AppletConnector appletConnector = null;
        boolean success = false;
        
        // get the AppletConnector object
        try {
            parentFrameAndFrameName = parentFrameAndFrameName.trim();
        } catch (NullPointerException e) {
            ErrorLog.log(this, "bringToFront", "The string \"parentFrameAndFrameName\" is null", e);
        }
        try {
            appletConnector = get (parentFrameAndFrameName);
            if (appletConnector != null) {
                // bring browser window to front
                appletConnector.sendToApplet("BringToFront", "", "");           // Bring window to front      
                success = true;
                return success;
            }
        } catch (Exception e) {
            ErrorLog.log(this, "bringToFront", "Tried to bring browser window to front, but the AppletConnector was null", e);
        }
        return false;

    }


    /**
     * This method calls JavaScript code inside the browser using the Applet to bring the browser's window behind the others
     *
     * @param parentFrameAndFrameName The name of the browserframe and, if existing, the framename of the parent frame to identify the right browser
     */
    public boolean blur(String parentFrameAndFrameName) {
        scone.accesstracking.AppletConnector appletConnector = null;
        boolean success = false;
        
        // get the AppletConnector object
        try {
            parentFrameAndFrameName = parentFrameAndFrameName.trim();
        } catch (NullPointerException e) {
            ErrorLog.log(this, "blur", "The string \"parentFrameAndFrameName\" is null", e);
        }
        try {
            appletConnector = get (parentFrameAndFrameName);
            if (appletConnector != null) {
                // bring browser window behind the others
                appletConnector.sendToApplet("BlurBrowser", "", "");           // blur browser window   
                success = true;
                return success;
            }
        } catch (Exception e) {
            ErrorLog.log(this, "blur", "Tried to bring browser window to front, but the AppletConnector was null", e);
        }
        return false;
    }


    /**
     * This method calls JavaScript code inside the browser using the Applet to resize the browser window
     *
     * @param parentFrameAndFrameName The name of the browserframe and, if existing, the framename of the parent frame to identify the right browser
     * @param width Width of the browser window
     * @param heigth Height of the browser window
     */
    public boolean resize(String parentFrameAndFrameName, int width, int height) {
        scone.accesstracking.AppletConnector appletConnector = null;
        boolean success = false;
        
        // get the AppletConnector object
        try {
            parentFrameAndFrameName = parentFrameAndFrameName.trim();
        } catch (NullPointerException e) {
            ErrorLog.log(this, "resize", "The string \"parentFrameAndFrameName\" is null", e);
        }
        try {
            appletConnector = get (parentFrameAndFrameName);
            if (appletConnector != null) {
                // resize browser window
                appletConnector.sendToApplet("AnyFunction", "window.resizeTo("+width+","+height+");", "");           // blur browser window   
                success = true;
                return success;
            }
        } catch (Exception e) {
            ErrorLog.log(this, "resize", "Tried to resize browser window, but the AppletConnector was null", e);
        }
        return false;
    }


    /**
     * This method calls JavaScript code inside the browser using the Applet to move the browser window
     *
     * @param parentFrameAndFrameName The name of the browserframe and, if existing, the framename of the parent frame to identify the right browser
     * @param x X-coordinate of left upper corner of the browser window
     * @param y Y-coordinate of left upper corner of the browser window
     */
    public boolean moveBrowserWindow(String parentFrameAndFrameName, int x, int y) {
        scone.accesstracking.AppletConnector appletConnector = null;
        boolean success = false;
        
        // get the AppletConnector object
        try {
            parentFrameAndFrameName = parentFrameAndFrameName.trim();
        } catch (NullPointerException e) {
            ErrorLog.log(this, "moveBrowserWindow", "The string \"parentFrameAndFrameName\" is null", e);
        }
        try {
            appletConnector = get (parentFrameAndFrameName);
            if (appletConnector != null) {
                // resize browser window
                appletConnector.sendToApplet("AnyFunction", "window.moveTo("+x+","+y+");", "");           // move browser window   
                success = true;
                return success;
            }
        } catch (Exception e) {
            ErrorLog.log(this, "moveBrowserWindow", "Tried to move browser window, but the AppletConnector was null", e);
        }
        return false;
    }


    /**
     * This method executes JavaScript code inside the browser using the Applet
     *
     * @param parentFrameAndFrameName The name of the browserframe and, if existing, the framename of the parent frame to identify the right browser
     * @param command javaScriptCommands to be executed inside the browser
     */
    public boolean javaScriptCommand(String parentFrameAndFrameName, String command) {
        scone.accesstracking.AppletConnector appletConnector = null;
        boolean success = false;
        
        // get the AppletConnector object
        try {
            parentFrameAndFrameName = parentFrameAndFrameName.trim();
        } catch (NullPointerException e) {
            ErrorLog.log(this, "javaScriptCommand", "The string \"parentFrameAndFrameName\" is null", e);
        }
        try {
            appletConnector = get (parentFrameAndFrameName);
            if (appletConnector != null) {
                // resize browser window
                appletConnector.sendToApplet("AnyFunction", command, "");           // blur browser window   
                success = true;
                return success;
            }
        } catch (Exception e) {
            ErrorLog.log(this, "javaScriptCommand", "Tried to send javaScript commands to the browser, but the AppletConnector was null", e);
        }
        return false;
    }


}
