/*
 * Scone - The Web Enhancement Framework
 * Copyright (c) 2004 Harald Weinreich, Volkert Buchmann, Frank Wollenweber, Torsten Ha
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package scone.usertesttool;


import java.awt.Color;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JComboBox;
import javax.swing.JLabel;


/**
 * UTComboBox is a ComboBox component for scone's user test tool. The ComboBox
 * is layed out with a label describing its contents. 
 * <p>Copyright: Copyright (c) 2004</p>
 * <p>Company: University of Hamburg, Germany</p>
 * @author Torsten Hass
 * @version 1.0, 08/27/2003
 */

public class UTComboBox extends Box implements InputComponent, AllComponents {
    private JComboBox   jComboBox;
    private String          compName = "";      // Name of this component, defined in test description xml file
    private ArrayList   itemList = null;
    private JLabel      descriptionLabel = null;
    private String      description = "";
    private long        timeStamp = 0;
    
    private UserTestControl utc;
    private UTComboBox thisInstance;
    
    // read out control
    int readOutCounter = 0;
    int numberOfData = 2;
    
    private boolean inputRequired = false;       // user has to choose a selectable item
    
    /**
     * Constructor with standard Size
     */
    public UTComboBox(UserTestControl utc) {
        this(utc, "", 1300); 
    }

    public UTComboBox(UserTestControl utc, String description) {
        this(utc, description, 1300); 
    }

    public UTComboBox(UserTestControl utc, String description, int width) {
        this(utc, description, width, true); 
    }

    /**
     * Set Heading, label and width.
     */
    public UTComboBox(UserTestControl utc, String description, int width, boolean enabled) {
        super(BoxLayout.Y_AXIS);
        itemList = new ArrayList();
        this.utc = utc;
        thisInstance = this;
        

        Box hBox = Box.createHorizontalBox();
        descriptionLabel = new JLabel(description);
        descriptionLabel.setFont(UIStyle.PLAINFONT);
        descriptionLabel.setForeground(Color.black);
        hBox.add(descriptionLabel);
        hBox.add(Box.createHorizontalGlue());
        add(hBox);
        
        hBox = Box.createHorizontalBox();

        jComboBox = new JComboBox();
        jComboBox.addActionListener(new Action2TimeStampActionListener());
        jComboBox.setPreferredSize(new Dimension(width, 22));
        jComboBox.setMaximumSize(new Dimension(width, 22));
        jComboBox.setEnabled(enabled);

        hBox.add(jComboBox);
        hBox.add(Box.createHorizontalGlue());
        add(hBox);
    }

    public void setComponentEnabled(boolean enable) {
        setEnabled(enable);
        jComboBox.setEnabled(enable);
        if (enable) {
            jComboBox.requestFocus();
        } else {   
            setComponentHighlighted(false);
        }
    }  

    /**
     * setName sets the name of the component. This name is defined in the
     * xml file as an attribute of the textField tag. This name is used to
     * destinct multiple text fields
     *
     * @param name The name for the text field
     */
/*    public void setName(String name) {
        this.name = name;
    }
    
    public String getName() {
        return name;
    }
*/    
    public void setInputRequired (boolean inputRequired) {
        this.inputRequired = inputRequired;
    }
    
    public void setDescription(String description) {
        this.description = description;
        descriptionLabel.setText(description);
    }
    
    /**
     * This method adds an item to the item list
     */
    public void addItem(long value, String text, boolean selectable, boolean showThisItemFirst) {
        itemList.add(new ComboItem(value, text, selectable));
        jComboBox.addItem(text);
        if (showThisItemFirst) {
            jComboBox.setSelectedIndex(jComboBox.getItemCount()-1);
        }
    }

     

    public void componentRequestFocus() {
        jComboBox.requestFocus();
    }

    public void setComponentHighlighted(boolean highlight) {
        if (highlight) {
            this.setBackground(new Color(230, 230, 128));
        } else {
            this.setBackground(new Color(204, 204, 204));
        }
    }


    /**
     * This method, called by UserTestControl, tests if an appropriate item
     * was selected. If inputRequired is true and the selected item is marked as
     * selectable = false then false is returned.
     */
    public boolean changedByUser () {
        if (inputRequired && !((ComboItem)itemList.get(jComboBox.getSelectedIndex())).isSelectable()) {
            jComboBox.requestFocus();
            return false;
        }
        return true;
    }
    
    public void setComponentName(String componentName) {
        compName = componentName;
    }
    
    public String getComponentName() {
        return compName;
    }
    
    // Data read out implementation

    /**
     * resets the data read out counter. The next read out data is the first available
     */
    public void startDataQuery() {
        readOutCounter = 0;
    }
    
    /**
     * returns true if there is data left, that was not read since startDataQuery()
     */
    public boolean hasMoreData() {
        return readOutCounter < numberOfData;
    }
    
    /**
     * Returns the name for the tag of the result file, the data should appear in 
     */
    public String getTagName() {
        switch (readOutCounter) {
            case 0: return "comboBoxText";
            case 1: return "comboBoxValue";
            default: return "";
        }
    }
    
    /**
     * Returns the gathered data, e.g. typed in text or the selected item of a comboBox
     */
    public String getData() {
        switch (readOutCounter) {
            case 0: return (String)jComboBox.getSelectedItem();
            case 1: return Long.toString(((ComboItem)itemList.get(jComboBox.getSelectedIndex())).getValue());
            default: return "";
        }
    }

    /**
     * Returns the timeStamp of the last modification by the user
     */
    public String getTimeStamp() {
        return Long.toString(timeStamp);
    }
    
    /**
     * prepares the next data set (if available) to be read out
     */
    public void getNextData() {
        if (readOutCounter < numberOfData) {
            readOutCounter++;
        }
    }
        
    class Action2TimeStampActionListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            timeStamp = System.currentTimeMillis();
        }
    }

    class ComboItem {
        private long value;
        private String text;
        private boolean selectable;
        
        
        public ComboItem (long value, String text, boolean selectable) {
            this.value = value;
            this.text = text;
            this.selectable = selectable;
        }
        
        
        public long getValue () {
            return value;
        }
    
    
        public boolean isSelectable () {
            return selectable;
        }
        
    }

    public void setComponentBottomPadding(int padding) {
    	// TODO do nothing
    }    
}
