/*
 * Scone - The Web Enhancement Framework
 * Copyright (c) 2004 Harald Weinreich, Volkert Buchmann, Frank Wollenweber, Torsten Ha
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package scone.usertesttool;


import java.awt.Color;
import java.awt.Dimension;
import java.util.ArrayList;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JSlider;
import javax.swing.JLabel;
import javax.swing.BorderFactory;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;


/**
 * UTSlider is a Slider component for scone's user test tool. The Slider
 * is layed out with a label describing its function. 
 * <p>Copyright: Copyright (c) 2004</p>
 * <p>Company: University of Hamburg, Germany</p>
 * @author Harald Weinreich
 * @version 1.0, 04/23/2004
 */

public class UTSlider extends Box implements InputComponent, AllComponents {

    private String      compName = "";      // Name of this component, defined in test description xml file
    private JSlider     jSlider;
    private int         bottomPadding = 10; // amount of vertical pixel added below the component
    private ArrayList   itemList = null;
    private String      description = "";
    private long        timeStamp = 0;
    private int         majorTickSpacing = 0;
    private int         minorTickSpacing = 0;
    private int         minimum          = 0;
    private int         maximum          = 100;
    private int         defaultValue     = 50;
    private boolean     snapToTicks      = false;
    private boolean     paintLabels      = false;
    
    private UserTestControl utc;
    private UTSlider thisInstance;
    
    private Box sliderBox;
    private Box paddingBox;
    
    // read out control
    int readOutCounter = 0;
    int numberOfData = 1;
    
    private boolean inputRequired = false;       // user has to choose a selectable item
    
    /**
     * Constructor with standard Size
     */
    public UTSlider(UserTestControl utc) {
        this(utc, "", 1300); 
    }

    public UTSlider(UserTestControl utc, String description) {
        this(utc, description, 1300); 
    }

    public UTSlider(UserTestControl utc, String description, int width) {
        this(utc, description, width, true); 
    }

    /**
     * Set Heading, label and width.
     */
    public UTSlider(UserTestControl utc, String description, int width, boolean enabled) {
        super(BoxLayout.Y_AXIS);
        itemList = new ArrayList();
        this.utc = utc;
        this.description = description;
        thisInstance = this;
        
        createSlider();
        
        setComponentEnabled(enabled);
    }

    /**
     * Show the slider
     */
    public void createSlider() {
        removeAll();        // clear Box in case of previous call of createLikertScale()
        
        sliderBox = Box.createHorizontalBox();
        sliderBox.add(Box.createHorizontalGlue());

        jSlider = new JSlider();
        jSlider.addChangeListener(new MyChangeListener());

        jSlider.setEnabled(true);
        jSlider.setBorder(BorderFactory.createTitledBorder(this.description));

        // Set borders
        jSlider.setMinimum(minimum);
        jSlider.setMaximum(maximum);
        jSlider.setValue(defaultValue);

        // Show ticks?            
        if (majorTickSpacing>0) {
            jSlider.setMajorTickSpacing(majorTickSpacing);
            jSlider.setPaintTicks(true);
        }
        if (minorTickSpacing>0) {
            jSlider.setMinorTickSpacing(minorTickSpacing);
            jSlider.setPaintTicks(true);
        }

        jSlider.setPaintLabels(paintLabels);
        jSlider.setSnapToTicks(snapToTicks);

        // set default size
        jSlider.setPreferredSize(new Dimension(400,80));
    
        sliderBox.add(jSlider);
        sliderBox.add(Box.createHorizontalGlue());
        add(sliderBox);

        paddingBox = new Box(BoxLayout.X_AXIS);
        paddingBox.add(Box.createRigidArea(new Dimension(0, bottomPadding)));
        add(paddingBox);
    }


    public void setComponentEnabled(boolean enable) {
        setEnabled(enable);
        jSlider.setEnabled(enable);
        if (enable) {
            jSlider.requestFocus();
        } else {   
            setComponentHighlighted(false);
        }
    }  

    /**
     * setName sets the name of the component. This name is defined in the
     * xml file as an attribute of the textField tag. This name is used to
     * destinct multiple text fields
     *
     * @param name The name for the text field
     */
/*    public void setName(String name) {
        this.name = name;
    }
    
    public String getName() {
        return name;
    }
*/    
    public void setInputRequired (boolean inputRequired) {
        this.inputRequired = inputRequired;
    }
    
    /**
     * This method adds the Description to the 
     */
    public void setDescription(String description) {
        this.description = description;
    }

    /**
     * This method adds major ticks to the slider
     */
    public void setMajorTickSpacing(int majorTickSpacing) {
        this.majorTickSpacing = majorTickSpacing;
    }

    /**
     * This method adds minor ticks to the slider
     */
    public void setMinorTickSpacing(int minorTickSpacing) {
        this.minorTickSpacing = minorTickSpacing;
    }

    /**
     * This method sets the minimum value
     */
    public void setMinimum(int minimum) {
        this.minimum = minimum;
    }

    /**
     * This method sets the default value of the slider
     */
    public void setDefault(int defaultValue) {
        this.defaultValue = defaultValue;
    }


    /**
     * This method sets the maximum value
     */
    public void setMaximum(int maximum) {
        this.maximum = maximum;
    }

    /**
     * Shall the slider snap to the displayed ticks?
     */
    public void setSnapToTicks(boolean snapToTicks) {
        this.snapToTicks = snapToTicks;
    }

    /**
     * Show the labels to the major ticks?
     */
    public void setPaintLabels(boolean paintLabels) {
        this.paintLabels = paintLabels;
    }



    public void componentRequestFocus() {
        jSlider.requestFocus();
    }

    public void setComponentHighlighted(boolean highlight) {
        if (highlight) {
            jSlider.setBackground(new Color(230, 230, 128));
        } else {
            jSlider.setBackground(new Color(204, 204, 204));
        }
    }


    /**
     * This method, called by UserTestControl, tests if an appropriate item
     * was selected. If inputRequired is true and the selected item is marked as
     * selectable = false then false is returned.
     */
    public boolean changedByUser () {
        if (inputRequired && !((ComboItem)itemList.get(jSlider.getValue())).isSelectable()) {
            jSlider.requestFocus();
            return false;
        }
        return true;
    }
    
    public void setComponentName(String componentName) {
        compName = componentName;
    }
    
    public void setComponentBottomPadding (int padding) {
        bottomPadding = padding;
        removeAll();
        paddingBox = new Box(BoxLayout.X_AXIS);
        paddingBox.add(Box.createRigidArea(new Dimension(0, bottomPadding)));
        
        add(sliderBox);
        add(paddingBox);
    }

    public String getComponentName() {
        return compName;
    }
    
    // Data read out implementation

    /**
     * resets the data read out counter. The next read out data is the first available
     */
    public void startDataQuery() {
        readOutCounter = 0;
    }
    
    /**
     * returns true if there is data left, that was not read since startDataQuery()
     */
    public boolean hasMoreData() {
        return readOutCounter < numberOfData;
    }
    
    /**
     * Returns the name for the tag of the result file, the data should appear in 
     */
    public String getTagName() {
        return "sliderValue";
    }
    
    /**
     * Returns the gathered data, e.g. typed in text or the selected item of a Slider
     */
    public String getData() {
        return Integer.toString(jSlider.getValue());
    }

    /**
     * Returns the timeStamp of the last modification by the user
     */
    public String getTimeStamp() {
        return Long.toString(timeStamp);
    }
    
    /**
     * prepares the next data set (if available) to be read out
     */
    public void getNextData() {
        if (readOutCounter < numberOfData) {
            readOutCounter++;
        }
    }
        
    class MyChangeListener implements ChangeListener {
        public void stateChanged(ChangeEvent e) {
            timeStamp = System.currentTimeMillis();
        }
    }

    class ComboItem {
        private long value;
        private String text;
        private boolean selectable;
        
        
        public ComboItem (long value, String text, boolean selectable) {
            this.value = value;
            this.text = text;
            this.selectable = selectable;
        }
        
        
        public long getValue () {
            return value;
        }
    
    
        public boolean isSelectable () {
            return selectable;
        }
        
    }
    
}
