/*
 * Scone - The Web Enhancement Framework
 * Copyright (C) 2009 Harald Weinreich, Volkert Buchmann, Frank Wollenweber, Torsten Ha
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
 package scone.util.tokenstream;


import adc.parser.HtmlStreamTokenizer;


/**
 * HTML text.
 * <p>This is unformattet HTML text
 *
 * @author Harald Weinreich
 * @author Volkert Buchmann 
 */

public class HtmlTextToken implements Token {
    public static final String COPYRIGHT = "(C) Harald Weinreich & Volkert Buchmann";
   	
    /**
     * preceding whitespaces
     */
    protected String whiteSpaces;
	
    /**
     * the text
     */
    protected String text;

    /**
     * the text without excaped characters like &amp;nbsp;
     */
    protected String unescapedText;

    /**
     * creates a <code>HtmlTextToken</code>
     *
     * @param text the text
     * @param whiteSpaces leading whitespaces (see the toString() method)
     */
    public HtmlTextToken(String text, StringBuffer whiteSpaces) {
        this.text = text;
        this.unescapedText = HtmlStreamTokenizer.unescape(text);
        this.whiteSpaces = new String(whiteSpaces);
    }
	
    /**
     * creates a <code>HtmlTextToken</code>
     *
     * @param text the text
     */
    public HtmlTextToken(String text) {
        this(text, new StringBuffer(""));
    }
	
    /**
     * returns a clone of this token
     * @return the clone
     */
    public Token getClone() {
        return new HtmlTextToken(text, new StringBuffer(whiteSpaces));
    }
	
    /**
     * Returns the simple text of the token without whitespaces.
     *
     * @return	a text string 
     */
    public String getText() {
        return text;
    }
   
    /**
     * Returns the token text with HTML escape sequences replaced with their character equivalents, e.g.
     * <b>&amp;copy;</b> becomes <b>&copy;</b>.
     *
     * @return	a string with all HTML escape sequences removed
     */
    public String getUnescapedText() {
        return unescapedText;
    }
   
    /**
     * changes the text
     * @param text the new text
     */
    public void setText(String text) {
        this.text = text;
    }
   
    /**
     * appends to the text
     * @param text the text to append
     */
    public void appendText(String text) {
        this.text += text;
    }
   
    /**
     * returns a String representation of this Token.
     * @return whiteSpaces+text
     */
    public String toString() {
        return whiteSpaces + text;
    }
	
    /**
     * returns the leading whitespaces
     * @return the whitespaces
     */
    public String getWhiteSpaces() {
        return whiteSpaces;
    }
	
    /**
     * changes the whitespaces
     * @param whiteSpaces the new whitespaces
     */
    public void setWhiteSpaces(String whiteSpaces) {
        this.whiteSpaces = whiteSpaces;
    }
}
